﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public class HistoryLogValidationCreate : IPlugin
    {
        private const string PluginName = "HistoryLogValidationCreate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            ITracingService tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_historylog")
                    return;

                OptionSetValue historyLogType = target.GetAttributeValue<OptionSetValue>("ppms_type");

                 //If we dont have a History Log Type, end logic. 
                if (historyLogType == null)
                return;

                //If this is not a Validation History Log Type we don't want to run any of the logic.
                if (historyLogType.Value != (int) PpmsHelper.HistoryLogType.Contact &&
                    historyLogType.Value != (int) PpmsHelper.HistoryLogType.Leie &&
                    historyLogType.Value != (int) PpmsHelper.HistoryLogType.License &&
                    historyLogType.Value != (int) PpmsHelper.HistoryLogType.NppesNpi)
                {
                    tracingService.Trace("Not a Validation History Log");
                    return; 
                }

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    tracingService.Trace("Retrieve provider related entities");
                    var providerRef = target.GetAttributeValue<EntityReference>("ppms_providerid");
                    Entity provider = GetProvider(service, providerRef.Id);
                    if (provider == null)
                    {
                        tracingService.Trace("Failed to retrieve provider");
                        return;
                    }

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    tracingService.Trace("Check History Log Validations");
                    var isComplete = IsValidationComplete(provider, service);

                    if (isComplete)
                    {
                        tracingService.Trace("Validations Complete");
                        var relatedbatchDetail = provider.RelatedEntities[new Relationship("ppms_account_batchdetail_provider")];

                        if (relatedbatchDetail != null)
                        {
                            // Get batch detail information
                            var batchDetail = relatedbatchDetail.Entities.FirstOrDefault();
                            //Batch Detail record may not always be present. 
                            if (batchDetail != null)
                            {
                                // Set to new status code
                                var batchState = (int)PpmsHelper.AccountState.Active;
                                var batchStatuscode = (int)PpmsHelper.Batch_StatusCode.ValidationComplete;
                                //Set Batch Detail State/Status
                                requests.Add(PpmsHelper.GetStateRequest(batchDetail.GetAttributeValue<EntityReference>("ppms_batch"), batchState, batchStatuscode));
                            }
                            else
                            {
                                tracingService.Trace("Batch detail not found");
                            }
                        }

                        //If we get here we have the Provider and Validations are complete. 
                        tracingService.Trace("Set Provider to Active");
                        var providerState = (int)PpmsHelper.AccountState.Active;
                        var providerStatus = (int)PpmsHelper.Account_StatusCode.Active;
                        //Set Provider State/Status
                        requests.Add(PpmsHelper.GetStateRequest(providerRef, providerState, providerStatus));

                        if (requests.Count > 0)
                        {
                            foreach (var request in requests)
                            {
                                service.Execute(request);
                            }
                        }
                        return;
                    }
                    tracingService.Trace("Validations Not Complete");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private IEnumerable<SetStateRequest> GetEntityStateRequests(Entity entity, string schemaName, int state, int statuscode)
        {
            List<SetStateRequest> requests = new List<SetStateRequest>();

            var children = entity.RelatedEntities[new Relationship(schemaName)];
            if (children != null)
            {
                foreach (var child in children.Entities)
                {
                    requests.Add(PpmsHelper.GetStateRequest(child, state, statuscode));
                }
            }

            return requests;
        }

        private Entity GetProvider(IOrganizationService service, Guid providerId)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("account", providerId);
            request.ColumnSet = new ColumnSet(new string[] { "accountid", "name", "statecode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_batchdetail_provider"),
                new QueryExpression("ppms_batchdetail")
                {
                    ColumnSet = new ColumnSet("ppms_batchdetailid", "ppms_name", "ppms_batch")
                }
            );

            //Get response
            var response = (RetrieveResponse)service.Execute(request);
            return response?.Entity;
        }

        private bool IsValidationComplete(Entity provider, IOrganizationService service)
        {
            using (var svc = new PpmsContext(service))
            {
                //Retrieve the Validation History Log types
                var validationHistoryLogs = svc.ppms_historylogSet
                            .Where(hl => hl.ppms_providerid.Id == provider.Id && (
                                         //hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Address ||
                                         //hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Geocode ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.License ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Sam)) 
                                         
                            .OrderByDescending(h => h.CreatedOn);

                //Add to list and check to make sure all the needed Validation Types are present

                //cehck the validationHistoryLogs for null / count
                if (validationHistoryLogs != null) { 
                    var validationHistoryLogsList = validationHistoryLogs.ToList();
                    //if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Address)) { return false; }
                    //if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Geocode)) { return false; }
                    if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact)) { return false; }
                    if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie)) { return false; }
                    if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.License)) { return false; }
                    if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi)) { return false; }
                    // TODO: Determine if SAM validation failure results in deactivation
                    //if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Sam)) { return false; }

                    //If all validations are present return true. 
                    return true; 
                }
                //If no validation logs are found return false. 
                return false;
            }            
        }
    }
}
